<?php

declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\ORM\Query;
use Cake\Database\StatementInterface;

class CountriesTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('countries');
		$this->setPrimaryKey('id');

		$this->hasMany('States', [
			'className' => 'App\Model\Table\StatesTable',
			'foreignKey' => 'country_id',
			'dependent' => true
		]);

		$this->hasOne('ShippingRates', [
			'className' => 'App\Model\Table\ShippingRatesTable',
			'foreignKey' => 'country_id',
			'dependent' => true
		]);
	}

	public function getDetails(int $id): array
	{
		$record = $this->find()
			->where(['id' => $id])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getCountryAgainstName(string $name): array
	{
		$record = $this->find()
			->where(['name' => $name])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getCountryAgainstShortName(string $name): array
	{
		$record = $this->find()
			->where(['short_name' => $name])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getCountriesAgainstKeyword(string $keyword): array
	{
		$connection = $this->getConnection();
		$stmt = $connection->execute("
			SELECT id, name, 'country' AS type FROM countries WHERE name LIKE :keyword
			UNION
			SELECT id, name, 'city' AS type FROM cities WHERE name LIKE :keyword
			UNION
			SELECT id, name, 'state' AS type FROM states WHERE name LIKE :keyword
			LIMIT 5
		", ['keyword' => $keyword . '%'], ['keyword' => 'string']);

		return $stmt instanceof StatementInterface ? $stmt->fetchAll('assoc') : [];
	}

	public function getAll(): array
	{
		return $this->find()
			->order(['name' => 'ASC'])
			->toArray();
	}

	public function setDefaultToZero(): void
	{
		$this->updateAll(['default' => 0], []);
	}
}
