<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\ORM\Query;
use DateTimeInterface;

class OrdersTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('order');
		$this->setPrimaryKey('id');
		$this->addBehavior('Timestamp');

		$this->belongsTo('Users', [
			'foreignKey' => 'user_id',
			'className' => 'Users',
		]);
		$this->belongsTo('DeliveryAddresses', [
			'foreignKey' => 'delivery_address_id',
			'className' => 'DeliveryAddresses',
		]);
		$this->belongsTo('Stores', [
			'foreignKey' => 'store_user_id',
			'className' => 'Users',
		]);
		$this->hasMany('OrderProducts', [
			'foreignKey' => 'order_id',
			'className' => 'OrderProducts',
		]);
		$this->hasOne('CouponUseds', [
			'foreignKey' => 'order_id',
			'className' => 'CouponUseds',
		]);
	}

	public function getDetails(int $id): array
	{
		$result = $this->find()
			->contain(['Users', 'DeliveryAddresses.Countries', 'OrderProducts', 'CouponUseds', 'Stores'])
			->where(['Order.id' => $id])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function getOrdersAccordingToStatus(int $status): array
	{
		$results = $this->find()
			->contain(['Users', 'DeliveryAddresses', 'OrderStoreProducts', 'CouponUseds', 'Stores.StoreLocations.Countries', 'Stores.Users'])
			->where(['Order.status' => $status])
			->all();

		return $results->toArray();
	}

	public function getTotalCoins(int $userId): float
	{
		$row = $this->find()
			->select(['total_amount' => 'SUM(total)'])
			->where(['status <' => 3, 'user_id' => $userId])
			->first();

		return (float)($row['total_amount'] ?? 0);
	}

	public function getPendingSellerBalance(int $userId, DateTimeInterface $olderDate): float
	{
		$row = $this->find()
			->select(['total_amount' => 'SUM(total)'])
			->where([
				'store_user_id' => $userId,
				'status' => 2,
				'completed_datetime !=' => '0000-00-00 00:00:00',
				'completed_datetime >=' => $olderDate->format('Y-m-d H:i:s'),
			])
			->first();

		return (float)($row['total_amount'] ?? 0);
	}

	public function getSellerBalance(int $userId, DateTimeInterface $olderDate): float
	{
		$row = $this->find()
			->select(['total_amount' => 'SUM(total)'])
			->where([
				'store_user_id' => $userId,
				'status' => 2,
				'completed_datetime !=' => '0000-00-00 00:00:00',
				'completed_datetime <=' => $olderDate->format('Y-m-d H:i:s'),
			])
			->first();

		return (float)($row['total_amount'] ?? 0);
	}

	public function getTotalCommissionEarned(int $userId): array
	{
		$row = $this->find()
			->select([
				'total_amount' => 'SUM(earned)',
				'total_orders' => 'COUNT(id)',
			])
			->innerJoinWith('Videos', function (Query $q) use ($userId) {
				return $q->where(['Videos.user_id' => $userId]);
			})
			->where(['video_id >' => 0, 'status' => 1, 'pickup' => 0])
			->first();

		return [
			'total_amount' => (float)($row['total_amount'] ?? 0),
			'total_orders' => (int)($row['total_orders'] ?? 0),
		];
	}

	public function getBalance(int $userId): float
	{
		$row = $this->find()
			->select(['total_amount' => 'SUM(subtotal)'])
			->matching('Stores', function (Query $q) use ($userId) {
				return $q->where(['Stores.user_id' => $userId]);
			})
			->where(['status' => 1])
			->first();

		return (float)($row['total_amount'] ?? 0);
	}

	public function getCountSoldItems(int $sellerId): int
	{
		return $this->find()
			->where(['store_user_id' => $sellerId])
			->count();
	}

	public function getCountStoreUserOrdersAccordingToStatus(int $status, int $userId): int
	{
		return $this->find()
			->where(['status' => $status, 'store_user_id' => $userId])
			->count();
	}

	public function getStoreUserOrdersAccordingToStatus(int $status, int $userId, int $startingId): array
	{
		$results = $this->find()
			->contain(['Users', 'DeliveryAddresses', 'OrderStoreProducts', 'CouponUseds', 'Stores.StoreLocations.Countries', 'Stores.Users'])
			->where(['status' => $status, 'store_user_id' => $userId])
			->limit(10)
			->offset($startingId * 10)
			->orderDesc('id')
			->all();

		return $results->toArray();
	}

	public function getStoreOrdersAccordingToStatus(int $status, int $storeId, int $startingId): array
	{
		$results = $this->find()
			->contain(['Users', 'DeliveryAddresses', 'OrderStoreProducts', 'CouponUseds', 'Stores.StoreLocations.Countries', 'Stores.Users'])
			->where(['status' => $status, 'store_id' => $storeId])
			->limit(10)
			->offset($startingId * 10)
			->all();

		return $results->toArray();
	}

	public function getStoreOrdersAccordingToStatusStorePortal(int $status, int $storeId): array
	{
		$results = $this->find()
			->contain(['Users', 'DeliveryAddresses', 'OrderStoreProducts', 'CouponUseds', 'Stores.StoreLocations.Countries', 'Stores.Users'])
			->where(['status' => $status, 'store_id' => $storeId])
			->all();

		return $results->toArray();
	}

	public function getUserOrders(int $userId, int $status, int $startingPoint): array
	{
		$query = $this->find()
			->contain(['OrderProducts', 'DeliveryAddresses', 'Stores'])
			->where(function ($exp) use ($userId, $status) {
				$conds = ['user_id' => $userId];
				if ($status !== 5) {
					$conds['status'] = $status;
				}
				return $exp->add($conds);
			})
			->limit(10)
			->offset($startingPoint * 10)
			->orderDesc('id');

		return $query->all()->toArray();
	}

	public function getStoreOrders(int $userId, int $status, int $startingPoint): array
	{
		$query = $this->find()
			->contain(['OrderProducts', 'Users', 'Stores'])
			->where(function ($exp) use ($userId, $status) {
				$conds = ['store_user_id' => $userId];
				if ($status !== 5) {
					$conds['status'] = $status;
				}
				return $exp->add($conds);
			})
			->limit(10)
			->offset($startingPoint * 10)
			->orderDesc('id');

		return $query->all()->toArray();
	}
}
