<?php
declare(strict_types=1);

namespace App\Datasource;

use Cake\Http\Client;
use Cake\Log\Log;
use Cake\Core\Exception\Exception;
use Cake\Utility\Hash;
use Cake\Model\Model;

class StripeSource extends \Cake\Datasource\DataSource
{
	protected Client $http;
	protected ?string $lastError = null;
	protected array $request = [];

	public function __construct(array $config = [])
	{
		parent::__construct($config);
		if (empty($config['api_key'])) {
			throw new Exception('StripeSource: Missing api key');
		}
		$this->http = new Client([
			'base_uri' => 'https://api.stripe.com/v1/',
			'auth' => [$config['api_key'], '']
		]);
	}

	public function create(Model $model, array $fields = [], array $values = []): bool
	{
		$body = $this->reformat($model, array_combine($fields, $values));
		$response = $this->request([
			'method' => 'POST',
			'uri' => ['path' => $model->path],
			'body' => $body
		]);
		if ($response === false) {
			return false;
		}
		$model->setInsertId($response['id']);
		$model->id = $response['id'];
		return true;
	}

	public function read(Model $model, array $queryData = [], ?int $recursive = null): mixed
	{
		if ($queryData['fields'] === 'COUNT') {
			return [[['count' => 1]]];
		}
		$pk = $model->alias . '.' . $model->primaryKey;
		if (empty($queryData['conditions'][$pk])) {
			$queryData['conditions'][$pk] = $model->id;
		}
		$path = trim($model->path, '/') . '/' . $queryData['conditions'][$pk];
		$response = $this->request(['uri' => ['path' => $path]]);
		if ($response === false) {
			return false;
		}
		$model->id = $response['id'];
		return [[ $model->alias => $response ]];
	}

	public function update(Model $model, array $fields = [], array $values = [], array $conditions = []): mixed
	{
		$data = array_combine($fields, $values);
		$id = $data['id'] ?? $model->id;
		unset($data['id']);
		$body = $this->reformat($model, $data);
		$path = trim($model->path, '/') . '/' . $id;
		$response = $this->request([
			'method' => 'POST',
			'uri' => ['path' => $path],
			'body' => $body
		]);
		if ($response === false) {
			return false;
		}
		$model->id = $id;
		return [$model->alias => $response];
	}

	public function delete(Model $model, array $conditions = []): bool
	{
		$pk = $model->alias . '.' . $model->primaryKey;
		$path = trim($model->path, '/') . '/' . $conditions[$pk];
		$response = $this->request([
			'method' => 'DELETE',
			'uri' => ['path' => $path]
		]);
		return $response !== false;
	}

	protected function request(array $request = []): mixed
	{
		$this->lastError = null;
		$method = $request['method'] ?? 'GET';
		$path = trim($request['uri']['path'] ?? '', '/');
		$body = $request['body'] ?? [];
		$response = match ($method) {
			'GET'    => $this->http->get($path),
			'POST'   => $this->http->post($path, $body),
			'DELETE' => $this->http->delete($path),
			default  => throw new Exception('Invalid HTTP method')
		};
		$status = $response->getStatusCode();
		$data = $response->getJson();
		if ($status === 200) {
			return $data;
		}
		if (in_array($status, [400, 402], true)) {
			$this->lastError = $data['error']['message'] ?? 'Stripe error';
			return false;
		}
		$this->lastError = 'Unexpected error.';
		Log::error($this->lastError);
		return false;
	}

	protected function reformat(Model $model, array $data): array
	{
		if (empty($model->formatFields)) {
			return $data;
		}
		foreach ($data as $field => $value) {
			foreach ($model->formatFields as $key => $fields) {
				if (in_array($field, $fields, true)) {
					$data[$key][$field] = $value;
					unset($data[$field]);
				}
			}
		}
		return $data;
	}

	public function calculate(Model $model, string $func): string
	{
		return 'COUNT';
	}

	public function listSources($data = null): ?array
	{
		return null;
	}

	public function describe(Model $model): ?array
	{
		return $model->_schema ?? null;
	}
}
