<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class VideoLikesTable extends Table
{
	public array $customFields = [
		"Videos.id",
		"Videos.user_id",
		"Videos.description",
		"Videos.video",
		"Videos.thum",
		"Videos.thum_small",
		"Videos.gif",
		"Videos.view",
		"Videos.sound_id",
		"Videos.privacy_type",
		"Videos.allow_comments",
		"Videos.allow_duet",
		"Videos.duration",
		"Videos.promote",
		"Videos.pin_comment_id",
		"Videos.pin",
		"Videos.location_string",
		"Videos.location_id",
		"Videos.lat",
		"Videos.long",
		"Videos.width",
		"Videos.height",
		"Videos.user_thumbnail",
		"Videos.default_thumbnail"
	];

	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('video_like');
		$this->setPrimaryKey('id');

		$this->belongsTo('Videos', [
			'foreignKey' => 'video_id',
		]);

		$this->belongsTo('Users', [
			'foreignKey' => 'user_id',
			'fields' => USER_FIELDS,
		]);
	}

	public function getDetails(int $id): ?array
	{
		return $this->find()
			->where(['VideoLikes.id' => $id, 'Videos.user_id >' => 0])
			->contain(['Videos.Users', 'Users'])
			->enableHydration(false)
			->first();
	}

	public function countLikes(int $video_id): int
	{
		return $this->find()
			->where(['VideoLikes.video_id' => $video_id])
			->count();
	}

	public function getUserVideoLikes(int $user_id): array
	{
		return $this->find()
			->where([
				'VideoLikes.user_id' => $user_id,
				'Videos.id IS NOT' => null
			])
			->contain(['Videos'])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function countLikesBetweenDatetime(array $video_ids, string $start_datetime, string $end_datetime): int
	{
		return $this->find()
			->where([
				'VideoLikes.video_id IN' => $video_ids,
				'VideoLikes.created >=' => $start_datetime,
				'VideoLikes.created <=' => $end_datetime
			])
			->count();
	}

	public function countLikesOnAllUserVideos(int $user_id): int
	{
		return $this->find()
			->matching('Videos', function ($q) use ($user_id) {
				return $q->where(['Videos.user_id' => $user_id]);
			})
			->count();
	}

	public function countUserAllVideoLikes(int $user_id): int
	{
		return $this->find()
			->where(['VideoLikes.user_id' => $user_id])
			->count();
	}

	public function getUserAllVideoLikes(int $user_id, int $starting_point): array
	{
		return $this->find()
			->where([
				'VideoLikes.user_id' => $user_id,
			])
			->matching('Videos', function ($q) {
				return $q->where(['Videos.id IS NOT' => null]);
			})
			->select($this->customFields)
			->limit(APP_RECORDS_PER_PAGE)
			->offset($starting_point * APP_RECORDS_PER_PAGE)
			->order(['VideoLikes.id' => 'DESC'])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function ifExist(array $data): ?array
	{
		return $this->find()
			->where([
				'VideoLikes.video_id' => $data['video_id'],
				'VideoLikes.user_id' => $data['user_id'],
				'Videos.user_id >' => 0
			])
			->contain(['Videos.Users', 'Users'])
			->enableHydration(false)
			->first();
	}

	public function getAll(): array
	{
		return $this->find()
			->enableHydration(false)
			->all()
			->toList();
	}
}
